<?php
/* --------------------------------------------------------------
   InfoPageContent.inc.php 2019-06-21
   Gambio GmbH
   http://www.gambio.de
   Copyright (c) 2019 Gambio GmbH
   Released under the GNU General Public License (Version 2)
   [http://www.gnu.org/licenses/gpl-2.0.html]
   --------------------------------------------------------------
*/

/**
 * Class InfoPageContent
 *
 * This class represents an info page content
 *
 * @category   System
 * @package    Content
 */
class InfoPageContent
    implements ContentAttributesInterface, ContentHeadingAttributeInterface, ContentTextAttributeInterface, ContentDownloadFileAttributeInterface, ContentMetaAttributesInterface, ContentNameAttributeInterface, ContentGroupIdInterface
{
    /**
     * "Allow Robots" status
     *
     * @var bool
     */
    protected $allowRobots;
    /**
     * Localized content download files
     *
     * @var ContentHeadingCollection
     */
    protected $downloads;
    /**
     * Localized content headings
     *
     * @var ContentHeadingCollection
     */
    protected $headings;
    /**
     * Content group ID
     *
     * @var ContentIdentificationInterface
     */
    protected $id;
    /**
     * Localized content meta descriptions
     *
     * @var ContentMetaDescriptionCollection
     */
    protected $metaDescriptions;
    /**
     * Localized content meta keywords
     *
     * @var ContentMetaKeywordsCollection
     */
    protected $metaKeywords;
    /**
     * Localized content meta titles
     *
     * @var ContentMetaTitleCollection
     */
    protected $metaTitles;
    /**
     * Localized content names
     *
     * @var ContentNameCollection
     */
    protected $names;
    /**
     * Localized content OpenGraph images
     *
     * @var ContentOpengraphImageCollection
     */
    protected $opengraphImages;
    /**
     * Content position
     *
     * @var string
     */
    protected $position;
    /**
     * Content site map information
     *
     * @var ContentSitemap
     */
    protected $sitemap;
    /**
     * Content status
     *
     * @var bool
     */
    protected $status;
    /**
     * Localized content texts
     *
     * @var ContentHeadingCollection
     */
    protected $texts;
    /**
     * Localized content titles
     *
     * @var ContentTitleCollection
     */
    protected $titles;
    /**
     * Content type
     *
     * @var string
     */
    protected $type;
    /**
     * Localized content URL keywords
     *
     * @var ContentUrlKeywordsCollection
     */
    protected $urlKeywords;
    /**
     * Localized content URL rewrites
     *
     * @var ContentUrlRewriteCollection
     */
    protected $urlRewrites;
    
    
    /**
     * InfoPageContent constructor.
     *
     * @param ContentPositionInterface              $pagePosition
     * @param ContentStatusInterface                $status
     * @param ContentTitleCollection                $titles
     * @param ContentHeadingCollection              $headings
     * @param ContentTextCollection                 $texts
     * @param ContentNameCollection                 $names
     * @param ContentIdentificationInterface|null            $id Optional content group ID
     * @param ContentDownloadFileCollection|null    $downloads
     * @param ContentMetaTitleCollection|null       $metaTitles
     * @param ContentMetaKeywordsCollection|null    $metaKeywords
     * @param ContentOpengraphImageCollection|null  $opengraphImages
     * @param ContentMetaDescriptionCollection|null $metaDescriptions
     * @param ContentAllowRobotsStatus|null         $allowRobotsStatus
     * @param ContentSitemap|null                   $sitemap
     * @param ContentUrlRewriteCollection|null      $urlRewrites
     * @param ContentUrlKeywordsCollection|null     $urlKeywords
     */
    public function __construct(
        ContentPositionInterface $pagePosition,
        ContentStatusInterface $status,
        ContentTitleCollection $titles,
        ContentHeadingCollection $headings,
        ContentTextCollection $texts,
        ContentNameCollection $names,
        ContentIdentificationInterface $id = null,
        ContentDownloadFileCollection $downloads = null,
        ContentMetaTitleCollection $metaTitles = null,
        ContentMetaKeywordsCollection $metaKeywords = null,
        ContentOpengraphImageCollection $opengraphImages = null,
        ContentMetaDescriptionCollection $metaDescriptions = null,
        ContentAllowRobotsStatus $allowRobotsStatus = null,
        ContentSitemap $sitemap = null,
        ContentUrlRewriteCollection $urlRewrites = null,
        ContentUrlKeywordsCollection $urlKeywords = null
    ) {
        $this->type = ContentType::createForContent()->type();
        //required
        $this->position = $pagePosition->position();
        $this->status   = $status->isActive();
        $this->titles   = $titles;
        $this->headings = $headings;
        $this->texts    = $texts;
        $this->names    = $names;
        //not required
        $this->id               = $id ?? null;
        $this->downloads        = $downloads;
        $this->metaTitles       = $metaTitles;
        $this->metaKeywords     = $metaKeywords;
        $this->opengraphImages  = $opengraphImages;
        $this->metaDescriptions = $metaDescriptions;
        $this->allowRobots      = $allowRobotsStatus ? $allowRobotsStatus->isAllowed() : null;
        $this->sitemap          = $sitemap;
        $this->urlRewrites      = $urlRewrites;
        $this->urlKeywords      = $urlKeywords;
    }
    
    
    /**
     * Return the content type
     *
     * @return string
     */
    public function type(): ?string
    {
        return $this->type;
    }
    
    
    /**
     * Return the content position
     *
     * @return string
     */
    public function position(): ?string
    {
        return $this->position;
    }
    
    
    /**
     * Return whether the content is active
     *
     * @return bool
     */
    public function isActive(): ?bool
    {
        return $this->status;
    }
    
    
    /**
     * Return the localized content titles
     *
     * @return ContentTitleCollection
     */
    public function titles(): ?ContentTitleCollection
    {
        return $this->titles;
    }
    
    
    /**
     * Return the localized content headings
     *
     * @return ContentHeadingCollection
     */
    public function headings(): ?ContentHeadingCollection
    {
        return $this->headings;
    }
    
    
    /**
     * Return the localized content texts
     *
     * @return ContentTextCollection
     */
    public function texts(): ?ContentTextCollection
    {
        return $this->texts;
    }
    
    
    /**
     * Return the localized content download files
     *
     * @return ContentDownloadFileCollection
     */
    public function downloads(): ?ContentDownloadFileCollection
    {
        return $this->downloads;
    }
    
    
    /**
     * Return the localized content meta titles
     *
     * @return ContentMetaTitleCollection
     */
    public function metaTitles(): ?ContentMetaTitleCollection
    {
        return $this->metaTitles;
    }
    
    
    /**
     * Return the localized content meta titles
     *
     * @return ContentMetaKeywordsCollection
     */
    public function metaKeywords(): ?ContentMetaKeywordsCollection
    {
        return $this->metaKeywords;
    }
    
    
    /**
     * Return the localized content meta descriptions
     *
     * @return ContentMetaDescriptionCollection
     */
    public function metaDescriptions(): ?ContentMetaDescriptionCollection
    {
        return $this->metaDescriptions;
    }
    
    
    /**
     * Return the localized content URL keywords
     *
     * @return ContentUrlKeywordsCollection
     */
    public function urlKeywords(): ?ContentUrlKeywordsCollection
    {
        return $this->urlKeywords;
    }
    
    
    /**
     * Return the localized content URL rewrites
     *
     * @return ContentUrlRewriteCollection
     */
    public function urlRewrites(): ?ContentUrlRewriteCollection
    {
        return $this->urlRewrites;
    }
    
    
    /**
     * Return the content site map information
     *
     * @return ContentSitemap
     */
    public function sitemap(): ?ContentSitemap
    {
        return $this->sitemap;
    }
    
    
    /**
     * Return whether robots are allowed
     *
     * @return bool
     */
    public function isAllowingRobots(): ?bool
    {
        return $this->allowRobots;
    }
    
    
    /**
     * Return the localized OpenGraph images
     *
     * @return ContentOpengraphImageCollection
     */
    public function opengraphImages(): ?ContentOpengraphImageCollection
    {
        return $this->opengraphImages;
    }
    
    
    /**
     * Return the localized content names
     *
     * @return ContentNameCollection
     */
    public function names(): ?ContentNameCollection
    {
        return $this->names;
    }
    
    
    /**
     * Returns the content group id if set, otherwise null will be returned.
     *
     * @return ContentIdentificationInterface|null
     */
    public function id() : ?ContentIdentificationInterface
    {
        return $this->id;
    }
}